function set_polarity_cap_interactively(cur_cell)
% all_bodies_uint8_cell, body_ind,  all_paths, params)

% polarity cap generator
% Created 29/10/2017
% Creates a polarity stack from a cell contour stack, by intersection with
% a sphere

%  bodies_cell;             % matrix for the shape iages
%  polarities_cell;         % matrix for the polarity images
%  xyzr;                    % coordinates of sphere used for cut-off of the polarity cap
%  all_paths;            % folder structure
%  img_ext;             % file format; used for reading of the shape stack and for saving of polarity stack
%  cortex_thickness;     % the cortex-thickness of polarity domain; pixels. Larger than 1; 2-3 is enough.
%  body_ind;

% conver uint8 to doubles first:
% polarity sub routine works with doubles; let's convert:

cur_cell.phi = 10;

uD                      = [];
uD.all_paths            = cur_cell.shape_paths;
uD.body_ind             = cur_cell.body_ind;
uD.shape_mat            = cur_cell.shape_mat;
uD.polarity_mat         = [];%zeros(size());
uD.cortex_thickness     = 3;%params.polar_domain_cortex_thickness;
uD.xyzr                 = [size(cur_cell.shape_mat, 2)/2, ...
                           size(cur_cell.shape_mat, 1)/2, ...
                           size(cur_cell.shape_mat, 3)/2, ...
                           20];
                       
[fig, xf, yf, zf, rf]   = create_main_figure(uD);

uD.x_field              = xf;
uD.y_field              = yf;
uD.z_field              = zf;
uD.r_field              = rf;

set(fig, 'UserData', uD);

update_main_fig(fig);

waitfor(fig);

end







function [fig, x_field, y_field, z_field, r_field] = create_main_figure(uD)

fieldH = 25;
fieldW = 50;

% Check plot of the output surface
screen_size         = get(0,'ScreenSize');
sw = screen_size(3);
sh = screen_size(4);
figW = 500;
figH = 500;
fx = (sw-figW)/2;
fy = (sh-figH)/2;

% create a new  figure:
fig = figure('position', [ fx, fy, figW, figH]);

hold on
title(['Body index: ' num2str(uD.body_ind)])

uicontrol('style', 'text', 'string', 'x', 'parent', fig, 'position',     [10,        figH-1*fieldH, fieldW, fieldH])
uicontrol('style', 'text', 'string', 'y', 'parent', fig, 'position',     [10,        figH-2*fieldH, fieldW, fieldH])
uicontrol('style', 'text', 'string', 'z', 'parent', fig, 'position',     [10,        figH-3*fieldH, fieldW, fieldH])
uicontrol('style', 'text', 'string', 'R', 'parent', fig, 'position',     [10,        figH-4*fieldH, fieldW, fieldH])
x_field = uicontrol('style', 'edit', 'string', num2str(uD.xyzr(1)), 'parent', fig, 'position',     [10+fieldW, figH-1*fieldH, fieldW, fieldH]);
y_field = uicontrol('style', 'edit', 'string', num2str(uD.xyzr(2)), 'parent', fig, 'position',     [10+fieldW, figH-2*fieldH, fieldW, fieldH]);
z_field = uicontrol('style', 'edit', 'string', num2str(uD.xyzr(3)), 'parent', fig, 'position',     [10+fieldW, figH-3*fieldH, fieldW, fieldH]);
r_field = uicontrol('style', 'edit', 'string', num2str(uD.xyzr(4)), 'parent', fig, 'position',    [10+fieldW, figH-4*fieldH, fieldW, fieldH]);
uicontrol('style', 'pushbutton', 'string', 'cut', 'parent', fig, 'position',    [10, figH-5*fieldH, 2*fieldW, fieldH], 'callback', @callback_test)
uicontrol('style', 'pushbutton', 'string', 'save', 'parent', fig, 'position',    [10, figH-6*fieldH, 2*fieldW, fieldH], 'callback', @callback_save)
uicontrol('style', 'pushbutton', 'string', 'quit', 'parent', fig, 'position',    [10, figH-7*fieldH, 2*fieldW, fieldH], 'callback', @callback_quit)

view([-15, 15])
box on;
axis equal
% set(gca, 'YDir', 'reverse');

xlabel('x');ylabel('y');zlabel('z');
xlim([0, size(uD.shape_mat, 2)]);     % x = columns
ylim([0 size(uD.shape_mat, 1)]);      % y = rows
zlim([0 size(uD.shape_mat, 3)]);      % z = 3rd dimension


end



function update_main_fig(fig)

figure(fig)
uD = get(fig, 'UserData');

cla;

color_set = [0 1 1];

patch(isosurface(uD.shape_mat, 0.5), 'FaceColor', color_set(1, :), 'EdgeColor', 'none', 'FaceAlpha',0.15);
     


[X,Y,Z] = sphere(100);
X = uD.xyzr(4) * X + uD.xyzr(1);
Y = uD.xyzr(4) * Y + uD.xyzr(2);
Z = uD.xyzr(4) * Z + uD.xyzr(3);
nucl_patch = surf2patch(X,Y,Z);

cut_sphere_centre_col = [1 1 1];
cut_sphere_col = 0*[1 1 1];

patch(nucl_patch, 'FaceColor', cut_sphere_col, 'FaceAlpha',0.05, 'EdgeColor','none');
plot3(uD.xyzr(1), uD.xyzr(2), uD.xyzr(3), 'o', 'markerfacecolor', 0.3*cut_sphere_centre_col, 'markeredgecolor', 0.1*cut_sphere_centre_col)

legend_str = {'body', 'cutting sphere', 'cutting sphere center'};

if ~isempty(uD.polarity_mat)
    patch(isosurface(uD.polarity_mat, 0.001), 'FaceColor', 'r', 'EdgeColor', 'none', 'FaceAlpha',0.3);
    legend_str = [legend_str, {'polarity domain'}];
end

legend(legend_str, 'Position', [0.6550,    0.0150,    0.3380,    0.1000]);

end




function callback_test(btn, evt)

% get parameters:
fig = get(btn,'Parent');
uD = get(fig, 'UserData');

uD.xyzr(1) = str2double(get(uD.x_field, 'String'));
uD.xyzr(2) = str2double(get(uD.y_field, 'String'));
uD.xyzr(3) = str2double(get(uD.z_field, 'String'));
uD.xyzr(4) = str2double(get(uD.r_field, 'String'));


uD = update_polarity_matrix(uD);

set(fig, 'UserData', uD);
update_main_fig(fig);
end



function uD = update_polarity_matrix(uD)

disp('        Calculating...')
cort_th             = uD.cortex_thickness;
xyzr                = uD.xyzr;

% 3d matrix for shape
% IMPORTANT: 1-inside, 0 - outside the cell.
% IT IS DIFFERENT FROM EARLIER VERSIONS!.
body_mat            = uD.shape_mat;

% empty polarity matrix:
polarity_mat        = zeros(size(body_mat));
sx                  = size(polarity_mat, 2);
sy                  = size(polarity_mat, 1);
sz                  = size(polarity_mat, 3);

% populate it with FULL CORTEX, carved out of the CURRENT BODY.
% We use sort of the edge of the shape only.
% importantly,  pixels in the edge from  slice N and  pixels in the edge 
% form slice N+1 must have direct face-to-face connections along the edge.
% (6-connection and not of higher order). 
% if this is not done, the following parts of the program may fail:
% Microtubules search for polarity domain and may not notice it if
% pixels are connected not face-to-face but corner-to-corner.    
% For this overlap to exists we simply make the edges wider (to the interior of
% the cell, but not exterior). Wider edges easily overlap. 

% FIND CORTEX IN YX planes
for z = 1 : sz
    im_yx          = body_mat(:, :, z);
    im_yx_eroded   = imerode(im_yx, strel('disk', cort_th));
    polarity_mat(:, :, z)     =  im_yx - im_yx_eroded;    
end       
% FIND CORTEX IN ZY planes (this is to deal with surfaces almost
% flat in XY plane)
for x = 1 : sx
    im_zy          = reshape(body_mat(:, x, :), sy, sz);
    im_zy_eroded   = imerode(im_zy, strel('disk', cort_th));
    im_zy_cortex   = im_zy - im_zy_eroded;
    im_zy_cortex   = reshape(im_zy_cortex, sy, 1, sz);
    polarity_mat(:, x, :)     =  polarity_mat(:, x, :) | im_zy_cortex; 
end    
 % FIND CORTEX IN ZX planes
for y = 1 : sy
    im_zx          = reshape(body_mat(y, :, :), sx, sz);
    im_zx_eroded   = imerode(im_zx, strel('disk', cort_th));
    im_zx_cortex   = im_zx - im_zx_eroded;
    im_zx_cortex   = reshape(im_zx_cortex, 1, sx, sz);
    polarity_mat(y, :, :)     =  polarity_mat(y, :, :) | im_zx_cortex; 
end  


% DEBUG: see how the cortex look like.
if 0
    figure;
    for z = 40:100  
        cla
        %imagesc(~body_mat(:, :, z) + ~polarity_mat(:, :, z))
        imagesc(polarity_mat(:, :, z))
        title(['slice ' num2str(z)])
        waitforbuttonpress;
    end
    %figure;
    %imagesc(~body_mat(:, :, z))      
    %imagesc(~polarities_cell{i} (:, :, z) + ~polarities_cell{i} (:, :, z+1))
end


% Take only a fraction of FULL CORTEX; this will be the polarity cap
% mask, empty yet:
distMat = false(size(polarity_mat)); 

 % carefull here with X,Y,Z order!
[X, Y, Z] = meshgrid(1:size(distMat, 2),...
                     1:size(distMat, 1),...
                     1:size(distMat, 3)); 

% set distances shorter than R to "true"
distMat = (X - xyzr(1)).^2 + ...
          (Y - xyzr(2)).^2 + ...
          (Z - xyzr(3)).^2 < xyzr(4)^2;    

% polarity stack: not true (larger than R) are set to 0.
% Inside cell = 1; outside = 0.
polarity_mat(~distMat) = 0;

% set:
uD.polarity_mat = polarity_mat;

% DEBUG See how finally (after cropping) the cortex look like
if 0
    figure;
    body_ind = 1;
    for z = 50:80  
        cla
        imagesc(~bodies_cell{body_ind}(:, :, z) + ~polarities_cell{body_ind} (:, :, z))
        title(['slice ' num2str(z)])
        waitforbuttonpress;
    end
    %figure;
    %imagesc(~bodies_cell{i}(:, :, z))      
    %imagesc(~polarities_cell{i} (:, :, z) + ~polarities_cell{i} (:, :, z+1))
end


disp('        finished calculation.')


end





function callback_save(btn, evt)

global global_polarity_cap;

% get parameters:
fig     = get(btn,'Parent');
uD      = get(fig, 'UserData');

userAnswer_1 = questdlg({['Use the current polarity domain?']});


switch userAnswer_1 
    
    case 'Yes'                
        global_polarity_cap = uD.polarity_mat;
        close(fig);
        
    otherwise
        disp('        Polarity matrix is not set yet.')
end

end






function callback_quit(btn, evt)

global global_polarity_cap;

% get parameters:
fig     = get(btn,'Parent');

userAnswer_1 = questdlg({['Are you sure you dont want to set polarity?'],...
                          ['Pulling forces will consist only of the first term: F = L^3']});


switch userAnswer_1 
    
    case 'Yes'                
        global_polarity_cap = [];
        close(fig);
        
    otherwise
end

end